<?php
/**
 * Document: class.gravatar.php
 * Created on: 30-gen-2012, 15:53
 * Author : Nicola Mustone <mustone.nicola@gmail.com>
   		    per Your Inspiration Web [http://www.yourinspirationweb.com]
 * Description: Classe Gravatar. Sfrutta le API di Gravatar per utilizzare il servizio fornito da http://www.gravatar.com/
 *
 * @version 1.0
 */
class Gravatar {
	/**
	 * Specifica se utilizzare o no il protocollo HTTPS
	 *
	 * @var bool
	 * @since 1.0
	 */
	public $useSSL;
	
	/**
	 * Se vero, restituisce l'url dell'avatar all'interno di un tag <img />
	 *
	 * @var bool
	 * @since 1.0
	 */
	public $imgWrapper;
	
	/**
	 * Forza l'utilizzo dell'immagine di default.
	 *
	 * @var bool
	 * @since 1.0
	 */
	public $forceDef;
	
	/**
	 * Idirizzo email da recuperare.
	 *
	 * @var string
	 * @private
	 * @since 1.0
	 */
	private $_email;
	
	/**
	 * Dimensione dell'immagine da recuperare. Il risultato è sempre un quadrato.
	 *
	 * E' possibile specificare un numero compreso tra 1 e 512, ma molti utenti non utilizzano alte risoluzioni
	 * quindi è consigliato non impostare un valore troppo alto.
	 *
	 * @var int
	 * @private
	 * @since 1.0
	 */
	private $_size;
	
	/**
	 * Immagine di default da associare nel caso l'email non sia associata a Gravatar.
	 *
	 * Possibili valori sono:
	 * 1. 404 - Non viene mostrata nessun'immagine.
	 * 2. mm - Mistery Man. Viene mostrato un omino bianco su sfondo grigio.
	 * 3. identicon - Un disegno geometrico creato in base all'hash dell'email.
	 * 4. monsterid - L'immaginetta di un mostro.
	 * 5. wavatar - Un faccina generata automaticamente.
	 * 6. retro - Una faccina in 8bit
	 *
	 * @var string
	 * @private
	 * @since 1.0
	 * @link http://scott.sherrillmix.com/blog/blogger/wp_identicon/
	 * @link http://scott.sherrillmix.com/blog/blogger/wp_monsterid/
	 * @link http://www.shamusyoung.com/twentysidedtale/?p=1462
	 */
	private $_default;
	
	/**
	 * Specifica il rating dell'immagine da recuperare. Se non viene soddisfatto questo parametro
	 * viene restituita l'immagine di default.
	 *
	 * Possibili valori:
	 * 1. g - Immagine utilizzabile su ogni tipo di sito.
	 * 2. pg - Può contenere gesti volgari, individui vestiti in modo provocatorio, parole volgari o violenza.
	 * 3. r - Può contenere intensa violenza, blasfemia, nudità o utilizzo di droghe.
	 * 4. x - Può contenere immagini a sfondo sessuale o violenza estrema.
	 *
	 * @var string
	 * @private
	 * @since 1.0
	 */
	private $_rating;
	
	/**
	 * Parametri extra da aggiungere nel caso si richieda di ottenere un'immagine nel tag <img />
	 *
	 * @var string
	 * @private
	 * @since 1.0
	 */
	private $_extras;
	
	/**
	 * Gravatar::__construct(). Costruttore della classe Gravatar.
	 *
	 * Setta le variabili pubbliche e private ai valori di default e imposta l'email del quale recuperare l'immagine.
	 *
	 * @param string $email
	 * @since 1.0
	 */
	public function __construct($email = false) {
		$this->useSSL = false;
		$this->imgWrapper = false;
		$this->forceDef = false;
		
		$this->_size = 80;
		$this->_default = 'mm';
		$this->_rating = NULL;
		$this->_extras = '';
		
		if(!empty($email)) {
			$email = trim(strtolower($email));
			$email = $this->_isValidEmail($email) ? $email : false;
		}
		
		$this->_email = $email;
	}
	
	/**
	 * Gravatar::setEmail(). Imposta l'indirizzo email del quale recuperare l'immagine.
	 *
	 * @param string $email
	 * @return bool
	 * @since 1.0
	 */
	public function setEmail($email) {
		if(!empty($email)) {
			$email = trim(strtolower($email));
			$email = $this->_isValidEmail($email) ? $email : false;
			
			$this->_email = $email;
		}
		
		return (bool) $email;
	}
	
	/**
	 * Gravatar::setDefault(). Imposta l'immagine di default da restituire.
	 *
	 * @param string $d
	 * @return bool
	 * @since 1.0
	 */
	public function setDefault($d) {
		$allowed = array('mm', '404', 'identicon', 'monsterid', 'wavatar', 'retro');
		$d = trim(strtolower($d));
		
		if(in_array($d, $allowed)) {
			$this->_default = $d;
			return true;
		}
		
		return false;
	}
	
	/**
	 * Gravatar::setSize(). Imposta la dimensione dell'immagine da restituire.
	 *
	 * @param int $s
	 * @return bool
	 * @since 1.0
	 */
	public function setSize($s) {
		if(is_int($s)) {
			if($s > 0 AND $s < 513) {
				$this->_size = $s;
				return true;
			}
		}
		
		return false;
	}
	
	/**
	 * Gravatar::setRating(). Imposta il rating da rispettare per l'immagine da restituire.
	 *
	 * @param string $r
	 * @return bool
	 * @since 1.0
	 */
	public function setRating($r) {
		$allowed = array('g', 'pg', 'r', 'x');
		$r = trim(strtolower($r));
		
		if(in_array($r, $allowed)) {
			$this->_rating = $r;
			return true;
		}
		
		return false;
	}
	
	/**
	 * Gravatar::setExtras(). Imposta i parametri extra da associare al tag <img />.
	 *
	 * @param array $params Deve essere un array associativo.
	 * @return bool
	 * @since 1.0
	 */
	public function setExtras($params) {
		if(is_array($params)) {
			$extras = '';
			
			foreach($params as $key => $value) {
				$extras .= trim(strtolower($key)) . '="' . trim($value) .'" ';
			}
			
			$this->_extras = $extras;
			return true;
		}
		
		return false;
	}
	
	/**
	 * Gravatar::getGravatar(). Restituisce il link dell'immagine richiesta.
	 *
	 * @return bool|string
	 * @since 1.0
	 */
	public function getGravatar() {
		$url  = $this->useSSL ? 'https://' : 'http://';
		$url .= 'www.gravatar.com/avatar/';
		
		if(!$this->_email) {
			return false;
		}
		
		$url .= md5($this->_email);	
		$url .= '?s=' . $this->_size;
		$url .= '&d=' . $this->_default;
		
		if(!empty($this->_rating)) {
			$url .= '&r=' . $this->_rating;	
		}
		
		if($this->forceDef) {
			$url .= '&f=y';	
		}
		
		if($this->imgWrapper) {
			$gravatar  = '';
			$gravatar .= '<img src="' . $url .'" ' . $this->_extras . '/>';
			
			return $gravatar;
		} else {
			return $url;
		}
	}
	
	/**
	 * Gravatar::__toString(). Metodo magico per stampare la classe Gravatar.
	 *
	 * @return string
	 * @since 1.0
	 * @see getGravatar()
	 */
	public function __toString() {
		return $this->getGravatar();	
	}
	
	/**
	 * Controlla la validità di un indirizzo email
	 *
	 * @param string $email
	 * @return bool
	 * @since 1.0
	 */
	private function _isValidEmail($email) {
		return preg_match('/^[\w.-]+(\+[\w.-]+)*@\w+[\w.-]*?\.\w{2,4}$/', $email);
	}
}
?>